import pygame
from sys import  exit
from random import randint

# -------------------- 胜利画面（赢了会显示） -------------------------
def show_success_page(screen):
    # 把胜利图片放上去，图片会拉成屏幕大小
    success_image = pygame.image.load("images/success.png")
    success_image = pygame.transform.scale(success_image, screen.get_size())
    
    # 一直显示胜利画面，除非你把窗口关掉
    while True:
        screen.blit(success_image, (0, 0))  # 把胜利图贴在屏幕上
        pygame.display.update()  # 更新画面
        for event in pygame.event.get():
            if event.type == pygame.QUIT:  # 点右上角 X 就退出游戏
                pygame.quit()
                exit()

# -------------------- 开始前的介绍页面（按空格才能继续） -------------------------
def show_intro_page(screen, image_path):
    # 把介绍图片放到屏幕上
    intro_image = pygame.image.load(image_path)
    intro_image = pygame.transform.scale(intro_image, screen.get_size())

    # 等你按下空格才能继续
    waiting = True
    while waiting:
        screen.blit(intro_image, (0, 0))
        pygame.display.update()
        for event in pygame.event.get():
            if event.type == pygame.QUIT:  # 点 X 也可以退出
                pygame.quit()
                exit()
            elif event.type == pygame.KEYDOWN and event.key == pygame.K_SPACE:
                waiting = False  # 按了空格，就不等了，往下继续

# -------------------- 幼苗类（会长成水稻） -------------------------
class Seeding(pygame.sprite.Sprite):
    def __init__(self, screen):
        super(Seeding, self).__init__()
        self.screen = screen

        # 加载幼苗和水稻的图，变一下大小
        self.image_normal = pygame.image.load('images/seeding.png')
        self.image_saved = pygame.image.load('images/rice.png')
        seeding_size = (120, 150)  # 幼苗的尺寸
        rice_size = (190, 200)     # 成熟水稻的尺寸
        self.image_normal = pygame.transform.scale(self.image_normal, seeding_size)
        self.image_saved = pygame.transform.scale(self.image_saved, rice_size)
        self.image = self.image_normal  # 一开始就是幼苗

        # 把幼苗放在屏幕底部，左右随机
        self.rect = self.image.get_rect()
        self.screenRect = self.screen.get_rect()
        self.rect.x = randint(0, self.screenRect.width - self.rect.width)
        self.rect.bottom = self.screenRect.bottom

        self.saved = False  # 还没施肥
        self.saved_time = 0  # 施肥后记录时间

    def update(self):
        # 施肥后等 0.5 秒就让它消失
        if self.saved and pygame.time.get_ticks() - self.saved_time > 500:
            self.kill()  # 把它从画面上删掉

    def blit(self):
        self.screen.blit(self.image, self.rect)  # 把它画在屏幕上

    def mark_as_saved(self):
        self.saved = True
        self.saved_time = pygame.time.get_ticks()  # 记录一下时间
        self.image = self.image_saved  # 换成水稻图片

# -------------------- 无人机类（可以飞来飞去） -------------------------
class Drone():
    def __init__(self, screen):
        self.screen = screen
        # 加载 4 张无人机图片，做动画用
        self.frames = [
            pygame.image.load('images/player1.png'),
            pygame.image.load('images/player2.png'),
            pygame.image.load('images/player3.png'),
            pygame.image.load('images/player4.png')
        ]
        self.frames = [pygame.transform.scale(img, (120, 120)) for img in self.frames]
        self.frame_index = 0  # 当前是第几张
        self.frame_speed = 0.2  # 动画切换得有多快
        self.image = self.frames[int(self.frame_index)]

        # 设置无人机在屏幕上的位置
        self.rect = self.image.get_rect()
        self.screenRect = self.screen.get_rect()
        self.rect.centerx = self.screenRect.centerx
        self.rect.top = self.screenRect.top
        self.speed = 5  # 无人机的速度

        # 哪些方向要移动
        self.moveLeft = self.moveRight = self.moveUp = self.moveDown = False

    def blit(self):
        # 每次切换下一张图，就像动起来了
        self.frame_index += self.frame_speed
        if self.frame_index >= len(self.frames):
            self.frame_index = 0
        self.image = self.frames[int(self.frame_index)]
        self.screen.blit(self.image, self.rect)

    def update(self):
        # 看看你按了哪个方向键，就动一下
        if self.moveLeft and self.rect.left > 0:
            self.rect.centerx -= self.speed
        if self.moveRight and self.rect.right < self.screenRect.right:
            self.rect.centerx += self.speed
        if self.moveUp and self.rect.top > 0:
            self.rect.bottom -= self.speed
        if self.moveDown and self.rect.bottom < self.screenRect.bottom:
            self.rect.bottom += self.speed

# -------------------- 肥料类（像子弹一样往下掉） -------------------------
class Supply(pygame.sprite.Sprite):
    def __init__(self, screen, droneSurface):
        super(Supply, self).__init__()
        self.screen = screen
        self.image = pygame.image.load('images/bullet.png')
        supply_size = (30, 50)  # 肥料图标大小
        self.image = pygame.transform.scale(self.image, supply_size)
        self.rect = self.image.get_rect()
        self.screenRect = self.screen.get_rect()

        # 肥料从无人机下面出现
        self.rect.top = droneSurface.rect.bottom - 40
        self.rect.centerx = droneSurface.rect.centerx

        self.speed = 5  # 掉下来的速度
        self.moveRun = False  # 要不要掉下来
        self.moveFollow = True  # 跟不跟着无人机动
        self.visible = False  # 暂时没用

    def blit(self):
        self.screen.blit(self.image, self.rect)

    def update(self):
        # 肥料往下掉
        self.rect.top += self.speed
        if self.rect.top > self.screenRect.bottom:  # 掉出屏幕就删掉
            self.kill()

# -------------------- 按键监听（控制无人机和肥料） -------------------------
def EventHandler(droneSurface, supplySurface):
    for event in pygame.event.get():
        if event.type == pygame.QUIT:
            pygame.quit()
            exit()
        elif event.type == pygame.KEYDOWN:
            # 按方向键就飞
            if event.key == pygame.K_RIGHT:
                droneSurface.moveRight = True
            elif event.key == pygame.K_LEFT:
                droneSurface.moveLeft = True
            elif event.key == pygame.K_UP:
                droneSurface.moveUp = True
            elif event.key == pygame.K_DOWN:
                droneSurface.moveDown = True
            elif event.key == pygame.K_SPACE:
                supplySurface.moveRun = True
                supplySurface.moveFollow = False
        elif event.type == pygame.KEYUP:
            if event.key == pygame.K_RIGHT:
                droneSurface.moveRight = False
            elif event.key == pygame.K_LEFT:
                droneSurface.moveLeft = False
            elif event.key == pygame.K_UP:
                droneSurface.moveUp = False
            elif event.key == pygame.K_DOWN:
                droneSurface.moveDown = False
            elif event.key == pygame.K_SPACE:
                supplySurface.moveRun = False
                supplySurface.moveFollow = True

# -------------------- 游戏主函数（所有都从这里开始） -------------------------
def initMainWindow():
    pygame.init()
    pygame.mixer.init()
    SCREEN_WIDTH = 800
    SCREEN_HEIGHT = 600
    screen = pygame.display.set_mode((SCREEN_WIDTH, SCREEN_HEIGHT))
    pygame.display.set_caption("低空助农")
    clock = pygame.time.Clock()
    
    # 开始前先看两张图片（介绍）
    show_intro_page(screen, "images/scene1.png")
    show_intro_page(screen, "images/scene2.png")
    
    # 播放背景音乐
    pygame.mixer.music.load('bgm/bgm.mp3')
    # 循环播放
    pygame.mixer.music.play(-1)
    # 载入碰撞音效
    boom_music = pygame.mixer.Sound('bgm/boom.mp3')

    # 加载背景图
    background = pygame.image.load("images/background.png")
    background = pygame.transform.scale(background, (SCREEN_WIDTH, SCREEN_HEIGHT))

    # 创建所有要用到的东西
    droneSurface = Drone(screen)
    supplySurface = Supply(screen, droneSurface)
    supplies = pygame.sprite.Group()
    seedings = pygame.sprite.Group()

    # 分数和生命值
    health_value = 100
    score = 0
    # 设置游戏字体
    gameFont = pygame.font.SysFont('Microsoft YaHei', 36, True)
    screenRect = screen.get_rect()
    
    last_shot_time = pygame.time.get_ticks()
    cooldown = 300  # 发一次肥料要等多久
    seeding_timer = 0  # 上次种幼苗的时间

    # 游戏一直进行，除非你失败或胜利
    while True:
        #画背景
        screen.blit(background, (0, 0))
        #画无人机
        droneSurface.blit()
        #检测无人机和肥料
        EventHandler(droneSurface, supplySurface)
        droneSurface.update()

        now = pygame.time.get_ticks()

        # 按空格键才能发肥料，而且要等 cooldown
        if supplySurface.moveRun and now - last_shot_time > cooldown:
            newSupply = Supply(screen, droneSurface)
            supplies.add(newSupply)
            last_shot_time = now

        supplies.draw(screen)
        supplies.update()

        # 看肥料有没有碰到幼苗
        collisions = pygame.sprite.groupcollide(seedings, supplies, False, True)
        for s in collisions:
            if not s.saved:
                boom_music.play()  # 播放碰撞音
                s.mark_as_saved()  # 幼苗变水稻
                health_value += 10
                score += 10

        # 每 3 秒种一个幼苗
        if now - seeding_timer > 3000:
            health_value -= 10  # 不施肥就扣血
            newSeeding = Seeding(screen)
            seedings.add(newSeeding)
            seeding_timer = now

        # 把幼苗都画出来
        for s in seedings:
            s.update()
            s.blit()

        # 显示分数和生命值
        healthRender = gameFont.render("生命值: %s" % str(health_value), True, (255, 25, 127))
        scoreRender = gameFont.render("得分: %s" % str(score), True, (25, 215, 210))
        screen.blit(healthRender, screen.get_rect().topleft)
        screen.blit(scoreRender, (screenRect.right - 160, 0))

        # 赢了：分数 >= 100
        if score >= 100:
            show_success_page(screen)
            break

        # 输了：撞幼苗 或 生命值太低
        if pygame.sprite.spritecollide(droneSurface, seedings, False) or health_value < 10:
            screen.blit(background, (0, 0))
            gameOverRender = gameFont.render("任务失败！", True, (255, 0, 0))
            screen.blit(gameOverRender, [screenRect.width / 3 + 50, screenRect.centery - 80])
            screen.blit(healthRender, [screenRect.width / 3 + 50, screenRect.centery - 30])
            pygame.display.update()
            break

        pygame.display.update()
        clock.tick(60)  # 每秒最多 60 帧，不卡顿

    # 游戏结束后画面不变，等你手动关掉
    while True:
        pygame.display.update()
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                pygame.quit()
                exit()

# -------------------- 程序开始啦 -------------------------
initMainWindow()
